/*==============================================================================
MASTER FILE: Part 2 - Estimation
==============================================================================
Purpose:
Estimate key variables for analysis including deposit betas, costs, and 
franchise values

Scripts executed:
1. 2a_deposit_betas.do    - Estimate deposit betas
2. 2b_deposit_costs.do  - Estimate deposit costs
3. 2c_deposit_franchises.do - Compute deposit franchise values
4. 2d_mtm_assets.do     - Calculate mark-to-market asset losses
5. 2e_bank_values.do - Calculate bank value measures
6. Within master: merge prepared variables into an analysis ready dataset.

Input:  call_reports_prepared.dta (part 1); tenyearrates.dta (Fed funds rates), indices_price_oad.dta (Bloomberg indices data)
Output: analysis_ready_data.dta in data/clean/

Last updated: Aug 9, 2025
==============================================================================*/

display "=== PART 2: VARIABLE CONSTRUCTION ==="

// Execute estimation scripts
// This creates temporary datasets in the temp directory for each step
// Note: some of these scripts inherits globals defined in MASTER.do to vary some parameters for extensions.
do "$path_code/02_estimation/2a_deposit_betas.do"
do "$path_code/02_estimation/2b_deposit_costs.do"
do "$path_code/02_estimation/2c_deposit_franchises.do"

do "$path_code/02_estimation/2d_mtm_assets.do"
do "$path_code/02_estimation/2e_bank_values.do"

// Merge all constructed variables into a single dataset using period as key
use "$path_temp/deposit_betas${ext_suffix}.dta", clear
merge 1:1 rssdid period using "$path_temp/deposit_costs${ext_suffix}.dta", nogen keep(3)
merge 1:1 rssdid period using "$path_temp/mtm_assets${ext_suffix}.dta", nogen
merge 1:1 rssdid period using "$path_temp/deposit_franchises${ext_suffix}.dta", nogen
merge 1:1 rssdid period using "$path_temp/bank_values${ext_suffix}.dta", nogen

// Take stock of final sample of banks where estimation inputs are nonmissing
foreach v in beta_depexp beta_ins beta_unins cost_ins cost_unins cost_tot loss DI_value DU_value DF_value DU_value_trunc DI_value_trunc DF_value_trunc v_noDF v_0 v_1 {
	keep if !missing(`v')
}

// bring in a couple of additional variables from the base data
merge 1:1 rssdid yq using "$path_clean/call_reports_prepared.dta", keepusing(name assets_bill uninsuredsh_domdep) keep(master match) nogen
drop yq 

// this replaces the share of uninsured deposits by its capped value for the appropriate extension.
if ${uninsured_cap} < 1 {
 replace uninsuredsh_domdep = ${uninsured_cap} if uninsuredsh_domdep>${uninsured_cap} 
}

// Save the final dataset with all constructed variables in long format
save "$path_clean/analysis_ready_data${ext_suffix}.dta", replace

// Clean up temp datasets in temp directory
erase "$path_temp/deposit_betas${ext_suffix}.dta"
erase "$path_temp/deposit_costs${ext_suffix}.dta"
erase "$path_temp/mtm_assets${ext_suffix}.dta"
erase "$path_temp/deposit_franchises${ext_suffix}.dta"
erase "$path_temp/bank_values${ext_suffix}.dta"

display "Part 2 completed: Analysis variables constructed"
